<?php
/**
 * Whiteboard64 Pro functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Whiteboard64 Pro
 */

if ( ! function_exists( 'whiteboard64_pro_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function whiteboard64_pro_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Whiteboard64 Pro, use a find and replace
	 * to change 'whiteboard64-pro' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'whiteboard64-pro', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );


	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary', 'whiteboard64-pro' ),
	) );



	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );



	// Set up the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'whiteboard64-pro_custom_background_args', array(
		'default-color' => 'ffffff',
		'default-image' => '',
	) ) );




	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );



	// Add theme support for custom logo feature.
	add_theme_support( 'custom-logo', array(
	   	'height'      => 75,
	   	'width'       => 350,
	   	'header-text' => array( 'site-title', 'site-description' ),
		) 
	);

	add_editor_style( array( 'css/editor-style.css', whiteboard64_pro_fonts_url() ) );
}
endif;
add_action( 'after_setup_theme', 'whiteboard64_pro_setup' );



if ( ! function_exists( 'whiteboard64_pro_fonts_url' ) ) :
/**
 * Register Google fonts for whiteboard64 pro.
 * @since whiteboard64-pro 1.0
 * @return string Google fonts URL for the theme.
 */
function whiteboard64_pro_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';

	/*
	 * Translators: If there are characters in your language that are not supported
	 * by Open Sans, translate this to 'off'. Do not translate into your own language.
	 */
	if ( 'off' !== _x( 'on', 'Open Sans font: on or off', 'whiteboard64-pro' ) ) {
		$fonts[] = 'Titillium+Web:400,300,600';
	}

	/*
	 * Translators: To add an additional character subset specific to your language,
	 * translate this to 'greek', 'cyrillic', 'devanagari' or 'vietnamese'. Do not translate into your own language.
	 */
	$subset = _x( 'no-subset', 'Add new subset (greek, cyrillic, devanagari, vietnamese)', 'whiteboard64-pro' );

	if ( 'cyrillic' == $subset ) {
		$subsets .= ',cyrillic,cyrillic-ext';
	} elseif ( 'greek' == $subset ) {
		$subsets .= ',greek,greek-ext';
	} elseif ( 'devanagari' == $subset ) {
		$subsets .= ',devanagari';
	} elseif ( 'vietnamese' == $subset ) {
		$subsets .= ',vietnamese';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' =>  implode( '|', $fonts ) ,
			'subset' =>  $subsets ,
		), '//fonts.googleapis.com/css' );
	}

	return $fonts_url;
}
endif;



function whiteboard64_pro_the_custom_logo() {
   if ( function_exists( 'the_custom_logo' ) ) {
      the_custom_logo();
   }
}


add_filter( 'widget_tag_cloud_args', 'whiteboard64_pro_tag_cloud_args' );
function whiteboard64_pro_tag_cloud_args( $args ) {
	$args['number'] = 20; // Your extra arguments go here
	$args['largest'] = 13;
	$args['smallest'] = 13;
	$args['unit'] = 'px';
	return $args;
}



function whiteboard64_pro_custom_excerpt_length( $length ) {
    return 20;
}
add_filter( 'excerpt_length', 'whiteboard64_pro_custom_excerpt_length', 999 );




// ================ whiteboard64-pro POST EXCERPTS ================== //
function whiteboard64_pro_excerpt($limit) {
    $excerpt = explode(' ', get_the_excerpt(), $limit);
    if (count($excerpt)>=$limit) {
    array_pop($excerpt);
        $excerpt = implode(" ",$excerpt).'...';
    } else {
        $excerpt = implode(" ",$excerpt);
    } 
        $excerpt = preg_replace('`\[[^\]]*\]`','',$excerpt);
    return $excerpt;
}

function whiteboard64_pro_content($limit) {
    $content = explode(' ', get_the_content(), $limit);
    if (count($content)>=$limit) {
        array_pop($content);
        $content = implode(" ",$content).'...';
    } else {
        $content = implode(" ",$content);
    } 
        $content = preg_replace('/\[.+\]/','', $content);
        $content = apply_filters('the_content', $content); 
        $content = str_replace(']]>', ']]&gt;', $content);
    return $content;
}


// ================ whiteboard64-pro SOCIAL SHARE ================== //
if ( ! function_exists( 'whiteboard64_pro_social_share' ) ) :
function whiteboard64_pro_social_share() {
    global $post;
    echo ' 
        
        <li><a target="_blank" href="http://www.facebook.com/share.php?u=';the_permalink(); echo'&t=';the_title(); echo'"><i class="fa fa-facebook"></i> Share on FaceBook</a></li>
        <li><a target="_blank" href="http://twitter.com/home?status=';the_title(); echo' - ';echo home_url()."/?p=".$post->ID; echo'"><i class="fa fa-twitter"></i> Share on Twitter</a></li>
        <li><a target="_blank" href="http://www.google.com/bookmarks/mark?op=edit&bkmk=';the_permalink(); echo'&title=';the_title(); echo'"><i class="fa fa-google-plus"></i> Share on Google Plus</a></li>
        <li><a target="_blank" href="http://www.linkedin.com/shareArticle?mini=true&url=';the_permalink(); echo'&title=';the_title();echo'"><i class="fa fa-linkedin"></i> Share on LinkedIn</a></li>
        ';
}
endif;


// ================ whiteboard64-pro POST DATED ================== //
function whiteboard64_pro_time_ago() { 
    global $post; 
    $date = get_post_time('G', true, $post); 
    // Array of time period chunks
    $chunks = array(
        array( 60 * 60 * 24 * 365 , __( 'year', 'whiteboard64-pro' ), __( 'years', 'whiteboard64-pro' ) ),
        array( 60 * 60 * 24 * 30 , __( 'month', 'whiteboard64-pro' ), __( 'months', 'whiteboard64-pro' ) ),
        array( 60 * 60 * 24 * 7, __( 'week', 'whiteboard64-pro' ), __( 'weeks', 'whiteboard64-pro' ) ),
        array( 60 * 60 * 24 , __( 'day', 'whiteboard64-pro' ), __( 'days', 'whiteboard64-pro' ) ),
        array( 60 * 60 , __( 'hour', 'whiteboard64-pro' ), __( 'hours', 'whiteboard64-pro' ) ),
        array( 60 , __( 'minute', 'whiteboard64-pro' ), __( 'minutes', 'whiteboard64-pro' ) ),
        array( 1, __( 'second', 'whiteboard64-pro' ), __( 'seconds', 'whiteboard64-pro' ) )
    );
 
    if ( !is_numeric( $date ) ) {
        $time_chunks = explode( ':', str_replace( ' ', ':', $date ) );
        $date_chunks = explode( '-', str_replace( ' ', '-', $date ) );
        $date = gmmktime( (int)$time_chunks[1], (int)$time_chunks[2], (int)$time_chunks[3], (int)$date_chunks[1], (int)$date_chunks[2], (int)$date_chunks[0] );
    }
 
    $current_time = current_time( 'mysql', $gmt = 0 );
    $newer_date = strtotime( $current_time );
 
    // Difference in seconds
    $since = $newer_date - $date; 
    // Something went wrong with date calculation and we ended up with a negative date.
    if ( 0 > $since )
        return __( 'sometime', 'whiteboard64-pro' ); 
    /**
     * We only want to output one chunks of time here, eg:
     * x years
     * xx months
     * so there's only one bit of calculation below:
     */
 
    //Step one: the first chunk
    for ( $i = 0, $j = count($chunks); $i < $j; $i++) {
        $seconds = $chunks[$i][0]; 
        // Finding the biggest chunk (if the chunk fits, break)
        if ( ( $count = floor($since / $seconds) ) != 0 )
            break;
    } 
    // Set output var
    $output = ( 1 == $count ) ? '1 '. $chunks[$i][1] : $count . ' ' . $chunks[$i][2];
 
    if ( !(int)trim($output) ){
        $output = '0 ' . __( 'seconds', 'whiteboard64-pro' );
    } 
    $output .= __(' ago', 'whiteboard64-pro'); 
    return $output;
}
 
// Filter our whiteboard64_pro_time_ago() function into WP's the_time() function
add_filter('the_time', 'whiteboard64_pro_time_ago');




/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function whiteboard64_pro_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'whiteboard64_pro_content_width', 640 );
}
add_action( 'after_setup_theme', 'whiteboard64_pro_content_width', 0 );





/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function whiteboard64_pro_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'whiteboard64-pro' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add sidebar widgets here.', 'whiteboard64-pro' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Header-Ad-Block', 'whiteboard64-pro' ),
		'id'            => 'header-ad-block',
		'description'   => esc_html__( 'Add Header Ad Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="header-ad">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Homepage-Ad1-Block', 'whiteboard64-pro' ),
		'id'            => 'homepage-ad1-block',
		'description'   => esc_html__( 'Add Homepage Ad1 Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Homepage-Ad2-Block', 'whiteboard64-pro' ),
		'id'            => 'homepage-ad2-block',
		'description'   => esc_html__( 'Add Homepage Ad2 Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Homepage-Ad3-Block', 'whiteboard64-pro' ),
		'id'            => 'homepage-ad3-block',
		'description'   => esc_html__( 'Add Homepage Ad3 Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Homepage-Ad4-Block', 'whiteboard64-pro' ),
		'id'            => 'homepage-ad4-block',
		'description'   => esc_html__( 'Add Homepage Ad4 Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Homepage-Ad5-Block', 'whiteboard64-pro' ),
		'id'            => 'homepage-ad5-block',
		'description'   => esc_html__( 'Add Homepage Ad5 Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Homepage-Ad6-Block', 'whiteboard64-pro' ),
		'id'            => 'homepage-ad6-block',
		'description'   => esc_html__( 'Add Homepage Ad6 Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Innerpage1-Ad-Block', 'whiteboard64-pro' ),
		'id'            => 'innerpage1-ad-block',
		'description'   => esc_html__( 'Add Innerpage Before Contents Ad Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Innerpage2-Ad-Block', 'whiteboard64-pro' ),
		'id'            => 'innerpage2-ad-block',
		'description'   => esc_html__( 'Add Innerpage After Contents Ad Block widget here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="block wow fadeInUp" data-wow-duration="3s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer-Block', 'whiteboard64-pro' ),
		'id'            => 'footer-block',
		'description'   => esc_html__( 'Add footer block widgets here.', 'whiteboard64-pro' ),
		'before_widget' => '<div class="col-md-4 footers">',
		'after_widget'  => '</div>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
}
add_action( 'widgets_init', 'whiteboard64_pro_widgets_init' );





/**
 * Enqueue scripts and styles.
 */
function whiteboard64_pro_scripts() {
	wp_enqueue_style( 'whiteboard64-pro-style', get_stylesheet_uri() );
	wp_enqueue_style( 'bootstrap', get_template_directory_uri().'/css/bootstrap.css' );
	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/fonts/font-awesome.css', array(), '4.6.3' );
	wp_enqueue_style( 'animate', get_template_directory_uri() . '/css/animate.css', array(), '3.4.0' );
	wp_enqueue_style( 'whiteboard64-pro-fonts', whiteboard64_pro_fonts_url(), array(), null );

	wp_enqueue_script( 'jquery-bootstrap', get_template_directory_uri() . '/js/bootstrap.js', array('jquery'), '3.3.7', true );	
	wp_enqueue_script( 'jquery-smartmenus', get_template_directory_uri() . '/js/jquery.smartmenus.js', array('jquery'), '1.0.0', true );
	wp_enqueue_script( 'jquery-tinycarousel', get_template_directory_uri() . '/js/jquery.tinycarousel.js', array('jquery'), '1.1.2', true );
	wp_enqueue_script( 'jquery-wow', get_template_directory_uri() . '/js/wow.js', array('jquery'), '1.1.2', true );
	wp_enqueue_script( 'whiteboard64-pro-scripts', get_template_directory_uri() . '/js/scripts.js', array('jquery'), '1.0.0', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'whiteboard64_pro_scripts' );





/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
require get_template_directory() . '/inc/jetpack.php';
require get_template_directory() . '/inc/navwalker.php';